﻿
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using Hims.Api.Utilities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;    
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;


    /// <inheritdoc />
    [Authorize]
    [Route("api/referralDoctor")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ReferralDoctorController : BaseController
    {

        /// <summary>
        /// The referralDoctor services.
        /// </summary>
        private readonly IReferralDoctorService referralDoctorService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public ReferralDoctorController(IReferralDoctorService referralDoctorService, IAuditLogService auditLogServices)
        {
            this.referralDoctorService = referralDoctorService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The add referralDoctor.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - referralDoctor added successfully.
        /// - 409 - referralDoctor already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] ReferralDoctorModel model,[FromHeader] LocationHeader location)
        {
            //model = (ReferralDoctorModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(location.LocationId);
            var response = await this.referralDoctorService.AddAsync(model);        
            switch (response)
            {
                case -2:
                    return this.Conflict("Given Mobile number has already been exists with us.");
                case -1:
                    return this.Conflict("Given Doctor name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.Masters,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.Now,
                LogDescription = $"{model.CreatedByName} has been added Referral doctor {model.FirstName + " " + model.LastName}.",
                LocationId=model.LocationId
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);
        }

        /// <summary>
        /// The fetch referralDoctor.
        /// </summary>
        /// <param name="model">
        /// The referralDoctor filter model.
        /// </param>
        /// <returns>
        /// The list of referralDoctor.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of referralDoctor.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ReferralDoctorModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ReferralDoctorModel model, [FromHeader] LocationHeader location)
        {
            model.LocationId = Convert.ToInt32(location.LocationId);
            var referralDoctor= await this.referralDoctorService.FetchAsync(model);
            return referralDoctor == null ? this.ServerError() : this.Success(referralDoctor);
        }

        /// <summary>
        /// The update referralDoctor.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - referralDoctor updated successfully.
        /// - 409 - referralDoctor already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] ReferralDoctorModel model,[FromHeader] LocationHeader header)
        {
            
            var response = await this.referralDoctorService.UpdateAsync(model);
            switch (response)
            {
                case -2:
                    return this.Conflict("Given Mobile number has already been exists with us.");
                case -1:
                    return this.Conflict("Given Referral Doctor name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Masters,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.Now,
                LogDescription = $"{model.ModifiedByName} has been updated Referral Doctor {model.FirstName+" "+model.LastName}.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);
        }

        /// <summary>
        /// The modify status of referralDoctor.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - referralDoctor status updated successfully.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] ReferralDoctorModel model, [FromHeader] LocationHeader header)
        {
            
            var response = await this.referralDoctorService.ModifyStatusAsync(model.ReferralDoctorId, Convert.ToInt32(model.ModifiedBy), model.Active);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Masters,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.Now,
                LogDescription = $"{model.ModifiedByName} has been updated the  status  to Referral doctor  {model.FullName} {(model.Active ? "'Active'" : "'Inactive'")}.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ReferralDoctorModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                var referralDoctorName = await this.referralDoctorService.FindNameByReferralDoctorId(model.ReferralDoctorId);
                var response = await this.referralDoctorService.DeleteAsync(model.ReferralDoctorId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Masters,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.ModifiedByName} has been deleted Referral doctor {referralDoctorName}.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success(response);
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("doctor can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }


    }
}
